/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.2,0.277777)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// "texture inversion" by Micah 2015
// modified from "p6mm inversion" by Carlos Urea (https://www.shadertoy.com/view/MtjGz3#)
// License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.


// *******************************************************************
// global defs and params
//

const float
    sqr2     = 1.4142135623730950488016887242096980785696, // square root of 2
    sqr3     = 1.7320508075688772935274463415058723669428, // square root of 3.0
    sqr2_inv = 1.0/sqr2 ,
    sqr3_inv = 1.0/sqr3 ,
    l        = 0.025,       // scaling factor
    l_inv    = 1.0/l ,
    line_w   = 0.02 ,  // 0.015
    speedFactor = 1.0 ;


vec2
    center   = 0.5*iResolution.xy, // viewport center in DC
    invDirection = vec2(0.0,1.0);

float
    mind   = min(iResolution.x,iResolution.y);

// -------------------------------------------------------------------------------

float dist( vec2 p, vec2 v1, vec2 v2 )
{
 	vec2   s = v2-v1 ,
           n = normalize(vec2( s.y, -s.x )) ;
    return dot(p-v1,n) ;
}
// -------------------------------------------------------------------------------

float DistanceFunc( float d )
{

   return 1.0-smoothstep( line_w*0.85, line_w*1.15, d );
}

// -------------------------------------------------------------------------------

vec2 DCToNDC( vec2 p_dc )
{
    return l_inv*(p_dc - center)/mind ;
}
// --------------------------------------------------------------------

vec2 Inversion( vec2 p, vec2 cen )
{
   float secs = iGlobalTime*speedFactor ;
   vec2  vr   = p  -cen ;
   float r    = length( vr );

    return cen + normalize(vr)/(r*0.1)
              + secs/4.0*invDirection
              + 1.0*l*vec2( sin(secs/40.0), cos(secs/42.0) ) ;
}

// -------------------------------------------------------------------------------

void main ( void )
{
    const int n = 9 ;
    const float n_inv = 1.0/float(n) ;

    vec4 res = vec4( 0.0, 0.0, 0.0, 1.0 );

    vec2 mou = DCToNDC( vec2(0.5*iResolution.x,0.0) );
    if ( iMouse.w != 0.0 )
        mou  = DCToNDC( iMouse.xy  ) ;


    for (int ix = 0 ; ix < n ; ix += 1 )
    for (int iy = 0 ; iy < n ; iy += 1 )
    {
       float px = -0.5 + (0.5+float(ix))*n_inv,
             py = -0.5 + (0.5+float(iy))*n_inv ;

       vec2 pNDC = DCToNDC( gl_FragCoord.xy + vec2( px, py ) );

       vec2 pinv = Inversion( pNDC, mou ) ;

       res += texture2D(iChannel0, pinv);
    }

    gl_FragColor = n_inv*n_inv*res ;
}
// -------------------------------------------------------------------------------
